# Commandify - Universal Command Palette for Admin Dashboards

![WordPress Plugin](https://img.shields.io/badge/WordPress-6.2%2B-blue.svg)
![PHP Version](https://img.shields.io/badge/PHP-7.4%2B-purple.svg)
![License](https://img.shields.io/badge/License-GPLv2%2B-green.svg)

A lightning-fast WordPress command palette: install plugins, switch themes, manage content, and navigate with Cmd+K or Ctrl+K.

## Description

**Commandify** brings the power of command palette navigation to WordPress, transforming how you work with your admin dashboard. **Install plugins, switch themes, manage posts, and navigate anywhere** with just a few keystrokes - no more clicking through endless menus.

Inspired by modern productivity tools like VS Code and Spotlight, Commandify provides a **blazing-fast search interface** that appears instantly with `Cmd+K` (Mac) or `Ctrl+K` (Windows/Linux). Type what you need, and Commandify finds it.

**Perfect for:**
- Site administrators installing and managing plugins/themes
- Content creators managing multiple posts and pages
- Developers who prefer keyboard navigation over mouse clicks
- Agency owners managing multiple client WordPress sites
- Anyone who wants to work faster and more efficiently

### 🚀 Why Commandify?

**Manage Plugins & Themes Without Leaving Your Keyboard**
Install, activate, deactivate plugins and switch themes in seconds. No more navigating through multiple admin pages - everything you need is just a keystroke away.

**Search Anything Instantly**
Find and manage your posts, pages, media, and users with lightning-fast search. Commandify's intelligent algorithm shows exactly what you're looking for.

**Speed Up Your Workflow**
Navigate WordPress admin 10x faster with keyboard shortcuts. Install a plugin, switch theme, or find a post - all in under 3 seconds.

**Keyboard-First Design**
Keep your hands on the keyboard. From plugin installation to theme activation to post editing, do everything without touching your mouse.

**Zero Learning Curve**
Familiar command palette interface that feels natural if you've used VS Code, Spotlight, or Alfred. If not, you'll master it in minutes.

## Core Features

### 🎯 Navigation Commands
- **Quick Access to Admin Pages**: Dashboard, Posts, Pages, Media, Comments, Appearance, Plugins, Users, Settings
- **Custom Post Types**: Automatically detects and adds all your custom post types
- **Hierarchical Menus**: Access submenu items directly (e.g., "Plugins → Add New")
- **Smart Contextual Navigation**: Relevant commands appear based on your current screen

### 🔍 Search Commands
- **Search Posts**: Find any post by title or content with live search
- **Search Pages**: Instant page search with fuzzy matching
- **Search Users**: Quickly find user accounts
- **Search Media**: Locate media files in your library
- **Custom Post Type Search**: Automatically adds search for all post types

### ⚡ Quick Actions
- **Check for Updates**: See available WordPress, plugin, and theme updates
- **Clear Cache**: Flush WordPress caches instantly
- **Empty Trash**: Permanently delete all trashed items
- **Contextual Actions**: Edit, view, preview, trash posts directly from search results

### 🔌 Plugin Management
- **Search Plugins**: Find and install plugins from WordPress.org repository (10,000+ plugins)
- **Install Plugins**: Install any plugin directly from the command palette
- **Activate Plugins**: Activate installed plugins with one keystroke
- **Deactivate Plugins**: Deactivate active plugins instantly
- **View Plugin Details**: See ratings, install count, and descriptions
- **One-Click Workflow**: Search → Install → Activate in seconds

### 🎨 Theme Management
- **Search Themes**: Browse and find themes from WordPress.org repository
- **Install Themes**: Install any theme directly from the palette
- **Switch Themes**: Activate/switch themes with a single command
- **View Theme Details**: Check ratings, downloads, and preview screenshots
- **Instant Theme Switching**: Change your site's look in seconds

### ⚙️ User Preferences
- **Keyboard Shortcut Customization**: Change the default `Cmd+K` / `Ctrl+K` to your preference
- **Results Per Page**: Control how many search results display (10-100)
- **Debounce Delay**: Adjust search responsiveness (0-500ms)
- **Hidden Commands**: Hide commands you don't use

### 🎨 User Experience
- **Instant Search**: Zero lag with client-side filtering
- **Beautiful UI**: Clean, modern interface that matches WordPress design
- **Keyboard Navigation**: Full keyboard support - never touch your mouse
- **Toast Notifications**: Clear success/error messages
- **Loading States**: Visual feedback for all actions
- **RTL Support**: Works perfectly with right-to-left languages

## Installation

### Automatic Installation
1. Go to **Plugins → Add New** in your WordPress admin
2. Search for "Commandify"
3. Click **Install Now** and then **Activate**
4. Press `Cmd+K` (Mac) or `Ctrl+K` (Windows/Linux) to open the command palette

### Manual Installation
1. Download the plugin ZIP file
2. Go to **Plugins → Add New → Upload Plugin**
3. Choose the downloaded file and click **Install Now**
4. Activate the plugin
5. Press `Cmd+K` or `Ctrl+K` to start using Commandify

### First Steps
1. Press `Cmd+K` / `Ctrl+K` to open the command palette
2. Try these commands:
   - Type "Install Plugin" to search and install plugins
   - Type "Activate" to activate installed plugins
   - Type "Switch Theme" to change your theme
   - Type "Search Posts" to find your content
   - Type "Preferences" to customize settings
3. Use arrow keys to navigate, Enter to execute
4. Press `Esc` or click outside to close

## Frequently Asked Questions

### How do I open the command palette?

Press `Cmd+K` on Mac or `Ctrl+K` on Windows/Linux. You can customize this keyboard shortcut in the preferences.

### Can I search for my posts and pages?

Yes! Type the name of any post or page, and Commandify will find it instantly. You can then edit, view, or trash it directly from the search results.

### Does it work with custom post types?

Absolutely! Commandify automatically detects all your custom post types and adds navigation and search commands for them.

### Can I disable Commandify on certain pages?

The command palette appears across all WordPress admin pages. You can hide specific commands you don't use through the preferences system.

### Is there a Pro version?

Yes! The Pro version includes additional features like smart default views, favorite commands, dynamic command patterns, contextual post actions, and settings management through the command palette. [Learn more](https://wprigel.com/commandify-pricing/)

### Does it slow down my WordPress admin?

Not at all! Commandify is optimized for performance with client-side command caching and instant filtering. The palette opens in milliseconds.

### Can I change the keyboard shortcut?

Yes! Go to the command palette (Cmd+K), type "keyboard shortcut", and choose your preferred combination.

### Does it work with multisite?

Yes, Commandify works perfectly on WordPress multisite installations.

### Is it compatible with other plugins?

Yes! Commandify is built to work alongside any WordPress plugin. It automatically integrates with admin menus added by other plugins.

### Can I install plugins from the command palette?

Absolutely! Type "Install Plugin" or just "Install" and start typing the plugin name. Commandify will search WordPress.org's plugin repository in real-time. Select any plugin to see options to install and activate it - all without leaving the command palette.

### How do I activate or deactivate plugins?

Type "Activate" to see all inactive plugins, or type "Deactivate" to see active plugins. Select any plugin to activate or deactivate it instantly. You can also search for a specific plugin by name.

### Can I switch themes from the command palette?

Yes! Type "Switch Theme" or "Activate Theme" to see all installed themes. Select one to activate it immediately. You can also type "Install Theme" to search and install themes from WordPress.org.

### How do I install a theme?

Open the command palette (Cmd+K / Ctrl+K), type "Install Theme", then start typing the theme name. Commandify will search the WordPress.org theme directory in real-time. Select any theme to install and activate it.

## Screenshots

1. **Command Palette** - Main interface with instant search
2. **Install Plugins** - Search and install from WordPress.org repository
3. **Activate/Deactivate Plugins** - Manage plugins with keyboard shortcuts
4. **Switch Themes** - Browse and activate themes instantly
5. **Search Posts** - Find any post with live search results
6. **Navigation Commands** - Quick access to all admin pages
7. **Post Actions** - Edit, preview, view, trash posts from search
8. **User Preferences** - Customize keyboard shortcuts and settings

## Changelog

### 1.0.0 - October 28, 2025
* Initial release
* **Plugin Management**: Search, install, activate, and deactivate plugins from WordPress.org
* **Theme Management**: Search, install, and switch themes instantly
* **Navigation Commands**: Quick access to all WordPress admin pages and menus
* **Search Commands**: Find posts, pages, users, and media with live search
* **Post Actions**: Edit, preview, view, and trash posts from search results
* **Quick Actions**: Check updates, clear cache, empty trash
* **User Preferences**: Customize keyboard shortcuts, results per page, and debounce delay
* **Custom Post Types**: Automatic detection and integration
* **Contextual Filtering**: Smart commands that appear based on current screen
* **Keyboard-First UI**: Full keyboard navigation with beautiful, modern design
* **Performance Optimized**: Client-side command caching for instant search
* **Developer friendly API**: Developer can create custom commands easily.

## Upgrade Notice

### 1.0.0
Initial release of Commandify. Install and press Cmd+K / Ctrl+K to supercharge your WordPress workflow!

## Requirements

- **WordPress**: 6.2 or higher
- **PHP**: 7.4 or higher
- **Tested up to**: WordPress 6.8

## Development

### 🛠️ Setting Up for Development

#### Prerequisites
- Node.js 14+ and npm
- WordPress development environment
- PHP 7.4+
- Composer (optional, for development tools)

#### Setup Steps

1. **Clone the Repository**
```bash
cd wp-content/plugins/
git clone https://github.com/wprigel/commandify.git
cd commandify
```

2. **Install Dependencies**
```bash
# Install npm packages
npm install

# Install Composer dependencies (optional, for linting)
composer install
```

3. **Build Assets**
```bash
# Development build (with watch mode)
npm run start

# Production build (minified)
npm run build
```

4. **Activate the Plugin**
- Go to WordPress Admin → Plugins
- Activate "Commandify"
- Press `Cmd+K` / `Ctrl+K` to test

#### Available Scripts

```bash
npm run start      # Start development with watch mode
npm run build      # Build for production
npm run lint       # Run ESLint and fix issues
npm run lint:check # Check for linting errors without fixing
```

#### Development Tools

- **ESLint**: JavaScript linting with WordPress coding standards
- **PHPCS**: PHP CodeSniffer for WordPress coding standards
- **Webpack**: Asset bundling and compilation
- **React**: Frontend component library
- **Sass**: CSS preprocessing

#### File Structure

```
commandify/
├── assets/
│   ├── src/              # React source files
│   │   ├── components/   # React components
│   │   ├── styles/       # Sass stylesheets
│   │   └── index.js      # Entry point
│   └── build/            # Compiled assets
├── includes/
│   ├── core-commands/    # Core command definitions
│   ├── Registry.php      # Command registry
│   ├── RestApi.php       # REST API endpoints
│   └── Settings.php      # Settings management
├── pro/                  # Pro features (if applicable)
└── commandify.php        # Main plugin file
```

### 🎯 Registering Custom Commands

Commandify provides a simple and powerful API for registering custom commands. You can hook into the `commandify_register_commands` action to add your own commands.

#### Command Types

Commandify supports four main command types:

1. **navigation** - Links to admin pages or external URLs
2. **action** - Executes a callback function (via REST API)
3. **search** - Dynamic search commands with live results
4. **group** - Parent commands with sub-actions

---

### 📝 Example 1: Navigation Command

Navigation commands redirect users to specific admin pages or URLs.

```php
add_action( 'commandify_register_commands', function() {
    commandify_register_command([
        'id'          => 'my-custom-page',
        'title'       => __( 'Go to Custom Page', 'my-plugin' ),
        'description' => __( 'Navigate to my custom admin page', 'my-plugin' ),
        'type'        => 'navigation',
        'callback'    => admin_url( 'admin.php?page=my-custom-page' ),
        'icon'        => 'dashicons-admin-generic',
        'category'    => 'navigation',
        'keywords'    => [ 'custom', 'page', 'settings' ],
        'capability'  => 'manage_options',
        'priority'    => 10,
    ]);
});
```

**Key Properties:**
- `callback` - URL to navigate to
- `capability` - WordPress capability required to see this command
- `keywords` - Additional search terms

---

### 🚀 Example 2: Action Command

Action commands execute server-side PHP callbacks via the REST API.

```php
add_action( 'commandify_register_commands', function() {
    commandify_register_command([
        'id'              => 'clear-custom-cache',
        'title'           => __( 'Clear Custom Cache', 'my-plugin' ),
        'description'     => __( 'Clear my plugin\'s custom cache', 'my-plugin' ),
        'type'            => 'action',
        'callback'        => 'my_plugin_clear_cache_callback',
        'icon'            => 'dashicons-trash',
        'category'        => 'tools',
        'keywords'        => [ 'cache', 'clear', 'flush' ],
        'capability'      => 'manage_options',
        'priority'        => 20,
        'loading_message' => __( 'Clearing cache...', 'my-plugin' ),
        'confirm'         => [
            'title'        => __( 'Clear Cache?', 'my-plugin' ),
            'message'      => __( 'Are you sure you want to clear the cache?', 'my-plugin' ),
            'confirm_text' => __( 'Clear Cache', 'my-plugin' ),
            'cancel_text'  => __( 'Cancel', 'my-plugin' ),
        ],
    ]);
});

/**
 * Callback function for the action command
 * Must return an array with success status and message
 */
function my_plugin_clear_cache_callback() {
    // Perform your cache clearing logic
    delete_transient( 'my_plugin_cache_key' );
    wp_cache_flush();

    return [
        'success' => true,
        'message' => __( 'Cache cleared successfully!', 'my-plugin' ),
        'data'    => [
            'timestamp' => current_time( 'mysql' ),
            'items_cleared' => 150,
        ],
    ];
}
```

**Key Properties:**
- `callback` - Function name (string) or callable
- `loading_message` - Message shown during execution
- `confirm` - Optional confirmation dialog
- **Return format**: Must return `['success' => bool, 'message' => string]`

---

### 🔍 Example 3: Search Command

Search commands provide dynamic, real-time search results with sub-actions.

```php
add_action( 'commandify_register_commands', function() {
    commandify_register_command([
        'id'          => 'search-products',
        'title'       => __( 'Search Products', 'my-plugin' ),
        'description' => __( 'Search through WooCommerce products', 'my-plugin' ),
        'type'        => 'search',
        'callback'    => 'my_plugin_search_products_callback',
        'icon'        => 'dashicons-products',
        'category'    => 'search',
        'keywords'    => [ 'product', 'woocommerce', 'search', 'find' ],
        'capability'  => 'edit_products',
        'priority'    => 15,
        'placeholder' => __( 'Type product name...', 'my-plugin' ),
        'debounce'    => 300, // milliseconds
    ]);
});

/**
 * Search callback function
 * Receives WP_REST_Request with 'query' parameter
 * Must return array of command objects
 */
function my_plugin_search_products_callback( $request ) {
    $search_query = $request->get_param( 'query' );

    // Query products
    $products = wc_get_products([
        's'      => $search_query,
        'limit'  => 20,
        'status' => 'publish',
    ]);

    $results = [];

    foreach ( $products as $product ) {
        $results[] = [
            'id'          => 'product-' . $product->get_id(),
            'title'       => $product->get_name(),
            'description' => sprintf(
                __( 'SKU: %s | Price: %s', 'my-plugin' ),
                $product->get_sku(),
                wc_price( $product->get_price() )
            ),
            'icon'        => $product->get_image( 'thumbnail' ),
            'actions'     => [
                [
                    'id'       => 'edit-product-' . $product->get_id(),
                    'label'    => __( 'Edit', 'my-plugin' ),
                    'callback' => get_edit_post_link( $product->get_id(), 'raw' ),
                    'type'     => 'navigation',
                ],
                [
                    'id'       => 'view-product-' . $product->get_id(),
                    'label'    => __( 'View', 'my-plugin' ),
                    'callback' => get_permalink( $product->get_id() ),
                    'type'     => 'navigation',
                    'target'   => '_blank',
                ],
                [
                    'id'       => 'delete-product-' . $product->get_id(),
                    'label'    => __( 'Delete', 'my-plugin' ),
                    'callback' => 'my_plugin_delete_product_callback',
                    'type'     => 'action',
                    'params'   => [ 'product_id' => $product->get_id() ],
                ],
            ],
        ];
    }

    return $results;
}
```

**Key Properties:**
- `callback` - Function that receives `WP_REST_Request`
- `placeholder` - Search input placeholder text
- `debounce` - Delay before search triggers (milliseconds)
- `actions` - Array of sub-actions for each result
- **Return format**: Array of command objects with `actions`

---

### 📦 Example 4: Group Command with Sub-actions

Group commands display a list of sub-actions when selected.

```php
add_action( 'commandify_register_commands', function() {
    commandify_register_command([
        'id'          => 'my-plugin-tools',
        'title'       => __( 'My Plugin Tools', 'my-plugin' ),
        'description' => __( 'Quick access to plugin tools', 'my-plugin' ),
        'type'        => 'group',
        'icon'        => 'dashicons-admin-tools',
        'category'    => 'tools',
        'keywords'    => [ 'tools', 'utilities' ],
        'capability'  => 'manage_options',
        'priority'    => 10,
        'actions'     => [
            [
                'id'       => 'export-data',
                'label'    => __( 'Export Data', 'my-plugin' ),
                'callback' => 'my_plugin_export_data',
                'type'     => 'action',
            ],
            [
                'id'       => 'import-data',
                'label'    => __( 'Import Data', 'my-plugin' ),
                'callback' => 'my_plugin_import_data',
                'type'     => 'action',
            ],
            [
                'id'       => 'reset-settings',
                'label'    => __( 'Reset Settings', 'my-plugin' ),
                'callback' => 'my_plugin_reset_settings',
                'type'     => 'action',
                'confirm'  => [
                    'title'   => __( 'Reset Settings?', 'my-plugin' ),
                    'message' => __( 'This will reset all plugin settings.', 'my-plugin' ),
                ],
            ],
        ],
    ]);
});
```

---

### 🎨 Available Command Properties

| Property | Type | Required | Description |
|----------|------|----------|-------------|
| `id` | string | ✅ | Unique command identifier |
| `title` | string | ✅ | Command title displayed in palette |
| `description` | string | ❌ | Additional description text |
| `type` | string | ✅ | Command type: `navigation`, `action`, `search`, `group` |
| `callback` | string/callable | ✅ | URL (navigation) or function name (action/search) |
| `icon` | string | ❌ | Dashicon class or image URL |
| `category` | string | ❌ | Category for grouping: `navigation`, `search`, `tools` |
| `keywords` | array | ❌ | Additional search keywords |
| `capability` | string | ❌ | WordPress capability required (default: `read`) |
| `priority` | int | ❌ | Sort order (lower = higher priority) |
| `loading_message` | string | ❌ | Message shown during action execution |
| `confirm` | array | ❌ | Confirmation dialog config |
| `actions` | array | ❌ | Sub-actions (for search results or groups) |
| `placeholder` | string | ❌ | Search input placeholder (search commands) |
| `debounce` | int | ❌ | Search delay in milliseconds |
| `target` | string | ❌ | Link target: `_self` or `_blank` |

---

### 🔗 Helper Functions

```php
// Register a command
commandify_register_command( array $command );

// Get all registered commands
commandify_get_commands();

// Check if a command exists
commandify_command_exists( string $command_id );

// Remove a command
commandify_unregister_command( string $command_id );
```

---

### 📚 Best Practices

1. **Use Unique IDs**: Prefix your command IDs with your plugin slug
2. **Capability Checks**: Always set appropriate capabilities
3. **Internationalization**: Wrap all user-facing strings with `__()` or `_e()`
4. **Error Handling**: Return proper error messages in action callbacks
5. **Performance**: Use debounce for search commands to reduce server load
6. **Keywords**: Add relevant keywords to improve searchability
7. **Icons**: Use descriptive icons or product images for better UX

---

### 🐛 Debugging

Enable WordPress debug mode to see command registration errors:

```php
define( 'WP_DEBUG', true );
define( 'WP_DEBUG_LOG', true );
```

Check the `wp-content/debug.log` file for any Commandify-related errors.

## License

This plugin is licensed under the GPLv2 or later.
[License URI](http://www.gnu.org/licenses/gpl-2.0.html)

## Author

Created by [wpRigel](https://wprigel.com)

## Support

For support, feature requests, or bug reports, please visit our [support forum](https://wordpress.org/support/plugin/commandify/) or [contact us](https://wprigel.com/contact/).

---

**⭐ If you find Commandify useful, please consider leaving a review on [WordPress.org](https://wordpress.org/support/plugin/commandify/reviews/)!**
