<?php
namespace WpRigel\Commandify;

defined( 'ABSPATH' ) || exit;

class Settings {

	private static $instance = null;

	const OPTION_KEY     = 'commandify_settings';
	const USER_PREFS_KEY = 'commandify_user_preferences';

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	public function add_settings_page() {
		add_options_page(
			__( 'Commandify Settings', 'commandify' ),
			__( 'Commandify', 'commandify' ),
			'manage_options',
			'commandify-settings',
			array( $this, 'render_settings_page' )
		);
	}

	public function register_settings() {
		register_setting(
			'commandify_settings_group',
			self::OPTION_KEY,
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
			)
		);

		// General Settings Section.
		add_settings_section(
			'commandify_general_section',
			__( 'General Settings', 'commandify' ),
			array( $this, 'render_general_section' ),
			'commandify-settings'
		);

		add_settings_field(
			'enable_palette',
			__( 'Enable Command Palette', 'commandify' ),
			array( $this, 'render_checkbox_field' ),
			'commandify-settings',
			'commandify_general_section',
			array(
				'label_for'   => 'enable_palette',
				'description' => __( 'Enable or disable the command palette for all users', 'commandify' ),
				'default'     => true,
			)
		);

		add_settings_field(
			'keyboard_shortcut',
			__( 'Keyboard Shortcut', 'commandify' ),
			array( $this, 'render_shortcut_field' ),
			'commandify-settings',
			'commandify_general_section',
			array(
				'label_for'   => 'keyboard_shortcut',
				'description' => __( 'Default keyboard shortcut to open the command palette', 'commandify' ),
				'default'     => 'mod+k',
			)
		);

		// Allow Pro features to register their own settings.
		do_action( 'commandify_register_settings', $this );

		// Performance Settings Section.
		add_settings_section(
			'commandify_performance_section',
			__( 'Performance Settings', 'commandify' ),
			array( $this, 'render_performance_section' ),
			'commandify-settings'
		);

		add_settings_field(
			'results_per_page',
			__( 'Results Per Page', 'commandify' ),
			array( $this, 'render_number_field' ),
			'commandify-settings',
			'commandify_performance_section',
			array(
				'label_for'   => 'results_per_page',
				'description' => __( 'Maximum number of results to display at once', 'commandify' ),
				'default'     => 50,
				'min'         => 10,
				'max'         => 100,
			)
		);

		add_settings_field(
			'debounce_delay',
			__( 'Search Debounce Delay (ms)', 'commandify' ),
			array( $this, 'render_number_field' ),
			'commandify-settings',
			'commandify_performance_section',
			array(
				'label_for'   => 'debounce_delay',
				'description' => __( 'Delay before search starts while typing (milliseconds)', 'commandify' ),
				'default'     => 150,
				'min'         => 0,
				'max'         => 1000,
			)
		);

		add_settings_field(
			'cache_ttl',
			__( 'Cache Duration (hours)', 'commandify' ),
			array( $this, 'render_number_field' ),
			'commandify-settings',
			'commandify_performance_section',
			array(
				'label_for'   => 'cache_ttl',
				'description' => __( 'How long to cache commands and search results', 'commandify' ),
				'default'     => 1,
				'min'         => 1,
				'max'         => 24,
			)
		);
	}

	public function render_general_section() {
		echo '<p>' . esc_html__( 'Configure the basic behavior of the command palette.', 'commandify' ) . '</p>';
	}

	public function render_performance_section() {
		echo '<p>' . esc_html__( 'Optimize the performance and responsiveness of the command palette.', 'commandify' ) . '</p>';
	}

	public function render_checkbox_field( $args ) {
		$settings = $this->get_settings();
		$value    = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : $args['default'];
		?>
		<label>
			<input
				type="checkbox"
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="<?php echo esc_attr( self::OPTION_KEY ); ?>[<?php echo esc_attr( $args['label_for'] ); ?>]"
				value="1"
				<?php checked( $value, true ); ?>
			/>
			<?php echo esc_html( $args['description'] ); ?>
		</label>
		<?php
	}

	public function render_shortcut_field( $args ) {
		$settings = $this->get_settings();
		$value    = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : $args['default'];
		?>
		<input
			type="text"
			id="<?php echo esc_attr( $args['label_for'] ); ?>"
			name="<?php echo esc_attr( self::OPTION_KEY ); ?>[<?php echo esc_attr( $args['label_for'] ); ?>]"
			value="<?php echo esc_attr( $value ); ?>"
			class="regular-text"
			placeholder="mod+k"
		/>
		<p class="description">
			<?php echo esc_html( $args['description'] ); ?><br>
			<?php esc_html_e( 'Use "mod" for Cmd/Ctrl, "shift", "alt", or specific keys like "k", "p", etc.', 'commandify' ); ?>
		</p>
		<?php
	}

	public function render_number_field( $args ) {
		$settings = $this->get_settings();
		$value    = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : $args['default'];
		?>
		<input
			type="number"
			id="<?php echo esc_attr( $args['label_for'] ); ?>"
			name="<?php echo esc_attr( self::OPTION_KEY ); ?>[<?php echo esc_attr( $args['label_for'] ); ?>]"
			value="<?php echo esc_attr( $value ); ?>"
			min="<?php echo esc_attr( $args['min'] ); ?>"
			max="<?php echo esc_attr( $args['max'] ); ?>"
			class="small-text"
		/>
		<p class="description"><?php echo esc_html( $args['description'] ); ?></p>
		<?php
	}

	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle cache clear.
		if ( isset( $_POST['commandify_clear_cache'] ) && check_admin_referer( 'commandify_clear_cache' ) ) {
			Cache::clear();
			delete_transient( 'commandify_menu_data' );
			echo '<div class="notice notice-success is-dismissible"><p>' .
				esc_html__( 'Cache cleared successfully!', 'commandify' ) .
				'</p></div>';
		}
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<form method="post" action="options.php">
				<?php
				settings_fields( 'commandify_settings_group' );
				do_settings_sections( 'commandify-settings' );
				submit_button( __( 'Save Settings', 'commandify' ) );
				?>
			</form>

			<hr>

			<h2><?php esc_html_e( 'Cache Management', 'commandify' ); ?></h2>
			<p><?php esc_html_e( 'Clear all cached commands and search results.', 'commandify' ); ?></p>
			<form method="post" action="">
				<?php wp_nonce_field( 'commandify_clear_cache' ); ?>
				<button type="submit" name="commandify_clear_cache" class="button button-secondary">
					<?php esc_html_e( 'Clear Cache', 'commandify' ); ?>
				</button>
			</form>

			<hr>

			<h2><?php esc_html_e( 'User Preferences', 'commandify' ); ?></h2>
			<p><?php esc_html_e( 'Users can customize their own preferences by clicking their profile in the command palette.', 'commandify' ); ?></p>
			<p><strong><?php esc_html_e( 'Available user preferences:', 'commandify' ); ?></strong></p>
			<ul style="list-style: disc; margin-left: 2em;">
				<li><?php esc_html_e( 'Keyboard shortcut override', 'commandify' ); ?></li>
				<li><?php esc_html_e( 'Results per page', 'commandify' ); ?></li>
				<li><?php esc_html_e( 'Search debounce delay', 'commandify' ); ?></li>
				<li><?php esc_html_e( 'Hidden commands list', 'commandify' ); ?></li>
			</ul>
		</div>
		<?php
	}

	public function sanitize_settings( $input ) {
		$sanitized = array();

		$sanitized['enable_palette'] = isset( $input['enable_palette'] ) ? true : false;

		$sanitized['keyboard_shortcut'] = isset( $input['keyboard_shortcut'] )
			? sanitize_text_field( $input['keyboard_shortcut'] )
			: 'mod+k';

		$sanitized['results_per_page'] = isset( $input['results_per_page'] )
			? absint( $input['results_per_page'] )
			: 50;
		if ( $sanitized['results_per_page'] < 10 ) {
			$sanitized['results_per_page'] = 10;
		}
		if ( $sanitized['results_per_page'] > 100 ) {
			$sanitized['results_per_page'] = 100;
		}

		$sanitized['debounce_delay'] = isset( $input['debounce_delay'] )
			? absint( $input['debounce_delay'] )
			: 150;
		if ( $sanitized['debounce_delay'] < 0 ) {
			$sanitized['debounce_delay'] = 0;
		}
		if ( $sanitized['debounce_delay'] > 1000 ) {
			$sanitized['debounce_delay'] = 1000;
		}

		$sanitized['cache_ttl'] = isset( $input['cache_ttl'] )
			? absint( $input['cache_ttl'] )
			: 1;
		if ( $sanitized['cache_ttl'] < 1 ) {
			$sanitized['cache_ttl'] = 1;
		}
		if ( $sanitized['cache_ttl'] > 24 ) {
			$sanitized['cache_ttl'] = 24;
		}

		// Allow Pro features to sanitize their own settings.
		$sanitized = apply_filters( 'commandify_sanitize_settings', $sanitized, $input );

		return $sanitized;
	}

	public function get_settings() {
		$defaults = array(
			'enable_palette'    => true,
			'keyboard_shortcut' => 'mod+k',
			'results_per_page'  => 50,
			'debounce_delay'    => 150,
			'cache_ttl'         => 1,
		);

		// Allow Pro features to add their own defaults.
		$defaults = apply_filters( 'commandify_settings_defaults', $defaults );

		$settings = get_option( self::OPTION_KEY, array() );
		return wp_parse_args( $settings, $defaults );
	}

	// User Preferences Methods.
	public static function get_user_preferences( $user_id = null ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$defaults = array(
			'keyboard_shortcut' => null,
			'results_per_page'  => null,
			'debounce_delay'    => null,
			'hidden_commands'   => array(),
		);

		$prefs = get_user_meta( $user_id, self::USER_PREFS_KEY, true );

		if ( ! is_array( $prefs ) ) {
			$prefs = array();
		}

		return wp_parse_args( $prefs, $defaults );
	}

	public static function update_user_preferences( $preferences, $user_id = null ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$current_prefs = self::get_user_preferences( $user_id );

		// Sanitize.
		$sanitized = array();

		if ( isset( $preferences['keyboard_shortcut'] ) ) {
			$sanitized['keyboard_shortcut'] = '' === $preferences['keyboard_shortcut']
				? null
				: sanitize_text_field( $preferences['keyboard_shortcut'] );
		}

		if ( isset( $preferences['results_per_page'] ) ) {
			$value                         = absint( $preferences['results_per_page'] );
			$sanitized['results_per_page'] = ( $value >= 10 && $value <= 100 ) ? $value : null;
		}

		if ( isset( $preferences['debounce_delay'] ) ) {
			$value                       = absint( $preferences['debounce_delay'] );
			$sanitized['debounce_delay'] = ( $value >= 0 && $value <= 1000 ) ? $value : null;
		}

		if ( isset( $preferences['hidden_commands'] ) ) {
			$sanitized['hidden_commands'] = is_array( $preferences['hidden_commands'] )
				? array_map( 'sanitize_key', $preferences['hidden_commands'] )
				: array();
		}

		$updated_prefs = array_merge( $current_prefs, $sanitized );

		return update_user_meta( $user_id, self::USER_PREFS_KEY, $updated_prefs );
	}

	public static function get_effective_preferences( $user_id = null ) {
		$instance        = self::get_instance();
		$global_settings = $instance->get_settings();
		$user_prefs      = self::get_user_preferences( $user_id );

		return array(
			'keyboard_shortcut' => $user_prefs['keyboard_shortcut'] ?? $global_settings['keyboard_shortcut'],
			'results_per_page'  => $user_prefs['results_per_page'] ?? $global_settings['results_per_page'],
			'debounce_delay'    => $user_prefs['debounce_delay'] ?? $global_settings['debounce_delay'],
			'hidden_commands'   => $user_prefs['hidden_commands'] ?? array(),
		);
	}
}

