<?php
namespace WpRigel\Commandify;

defined( 'ABSPATH' ) || exit;

/**
 * Register plugin and theme commands
 * Hooks into commandify_register_commands at priority 35 (before installation at 50)
 */
add_action(
	'commandify_register_commands',
	function () {
		// Activate Plugins - Show all inactive plugins.
		commandify_register_command(
			array(
				'id'          => 'activate-plugins',
				'title'       => __( 'Activate Plugins', 'commandify' ),
				'description' => __( 'View and activate inactive plugins', 'commandify' ),
				'type'        => 'search',
				'callback'    => __NAMESPACE__ . '\\get_inactive_plugins_callback',
				'category'    => 'plugins',
				'icon'        => 'dashicons-admin-plugins',
				'placeholder' => __( 'Search inactive plugins...', 'commandify' ),
				'min_chars'   => 0,
				'capability'  => 'activate_plugins',
				'keywords'    => array( 'activate', 'enable', 'plugin' ),
				'priority'    => 40,
			)
		);

		// Deactivate Plugins - Show all active plugins.
		commandify_register_command(
			array(
				'id'          => 'deactivate-plugins',
				'title'       => __( 'Deactivate Plugins', 'commandify' ),
				'description' => __( 'View and deactivate active plugins', 'commandify' ),
				'type'        => 'search',
				'callback'    => __NAMESPACE__ . '\\get_active_plugins_callback',
				'category'    => 'plugins',
				'icon'        => 'dashicons-admin-plugins',
				'placeholder' => __( 'Search active plugins...', 'commandify' ),
				'min_chars'   => 0,
				'capability'  => 'activate_plugins',
				'keywords'    => array( 'deactivate', 'disable', 'plugin' ),
				'priority'    => 35,
			)
		);

		// Switch Theme - Show all available themes.
		commandify_register_command(
			array(
				'id'          => 'switch-theme',
				'title'       => __( 'Switch Theme', 'commandify' ),
				'description' => __( 'Browse and activate themes', 'commandify' ),
				'type'        => 'search',
				'callback'    => __NAMESPACE__ . '\\get_available_themes_callback',
				'category'    => 'themes',
				'icon'        => 'dashicons-admin-appearance',
				'placeholder' => __( 'Search themes...', 'commandify' ),
				'min_chars'   => 0,
				'capability'  => 'switch_themes',
				'keywords'    => array( 'switch', 'change', 'theme', 'appearance' ),
				'priority'    => 30,
			)
		);
	}
);

/**
 * Get inactive plugins
 */
function get_inactive_plugins_callback( $query ) {
	if ( ! function_exists( 'get_plugins' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	if ( ! function_exists( 'is_plugin_active' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}

	$all_plugins = get_plugins();
	$results     = array();
	$query_lower = strtolower( trim( $query ) );

	foreach ( $all_plugins as $plugin_file => $plugin_data ) {
		// Skip active plugins.
		if ( is_plugin_active( $plugin_file ) ) {
			continue;
		}

		// Filter by search query if provided (only if user has typed something).
		if ( ! empty( $query_lower ) ) {
			$name_lower = strtolower( $plugin_data['Name'] );
			$desc_lower = strtolower( $plugin_data['Description'] );

			if ( strpos( $name_lower, $query_lower ) === false && strpos( $desc_lower, $query_lower ) === false ) {
				continue;
			}
		}

		$slug = dirname( $plugin_file );
		if ( '.' === $slug ) {
			$slug = basename( $plugin_file, '.php' );
		}

		// Decode any HTML entities in plugin name for display.
		$plugin_name = html_entity_decode( $plugin_data['Name'], ENT_QUOTES, get_bloginfo( 'charset' ) );

		$results[] = array(
			'id'                   => 'activate-plugin-' . $slug,
			'title'                => $plugin_name,
			'description'          => sprintf(
				/* translators: 1: Version number, 2: Author name */
				__( 'Version %1$s • %2$s', 'commandify' ),
				$plugin_data['Version'],
				$plugin_data['Author']
			),
			'type'                 => 'action',
			'icon'                 => 'dashicons-admin-plugins',
			'category'             => 'plugins',
			'has_dynamic_callback' => true,
			'loading_message'      => sprintf(
				/* translators: %s: Plugin name or theme name */
				__( 'Activating %s...', 'commandify' ),
				$plugin_data['Name']
			),
		);
	}//end foreach

	return $results;
}

/**
 * Get active plugins
 */
function get_active_plugins_callback( $query ) {
	if ( ! function_exists( 'get_plugins' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	if ( ! function_exists( 'is_plugin_active' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}

	$all_plugins = get_plugins();
	$results     = array();
	$query_lower = strtolower( trim( $query ) );

	foreach ( $all_plugins as $plugin_file => $plugin_data ) {
		// Skip inactive plugins.
		if ( ! is_plugin_active( $plugin_file ) ) {
			continue;
		}

		// Filter by search query if provided (only if user has typed something).
		if ( ! empty( $query_lower ) ) {
			$name_lower = strtolower( $plugin_data['Name'] );
			$desc_lower = strtolower( $plugin_data['Description'] );

			if ( strpos( $name_lower, $query_lower ) === false && strpos( $desc_lower, $query_lower ) === false ) {
				continue;
			}
		}

		$slug = dirname( $plugin_file );
		if ( '.' === $slug ) {
			$slug = basename( $plugin_file, '.php' );
		}

		// Don't allow deactivating Commandify itself.
		$is_commandify = ( strpos( $plugin_file, 'commandify' ) === 0 );

		// Skip Commandify itself for deactivation.
		if ( $is_commandify ) {
			continue;
		}

		// Decode any HTML entities in plugin name for display.
		$plugin_name = html_entity_decode( $plugin_data['Name'], ENT_QUOTES, get_bloginfo( 'charset' ) );

		$results[] = array(
			'id'                   => 'deactivate-plugin-' . $slug,
			'title'                => $plugin_name,
			'description'          => sprintf(
				/* translators: 1: Version number, 2: Author name */
				__( 'Version %1$s • %2$s', 'commandify' ),
				$plugin_data['Version'],
				$plugin_data['Author']
			),
			'type'                 => 'action',
			'icon'                 => 'dashicons-admin-plugins',
			'category'             => 'plugins',
			'has_dynamic_callback' => true,
			/* translators: %s: Plugin name or theme name */
			'loading_message'      => sprintf( __( 'Deactivating %s...', 'commandify' ), $plugin_data['Name'] ),
		);
	}//end foreach

	return $results;
}

/**
 * Get available themes
 */
function get_available_themes_callback( $query ) {
	$all_themes    = wp_get_themes();
	$current_theme = get_stylesheet();
	$results       = array();
	$query_lower   = strtolower( trim( $query ) );

	foreach ( $all_themes as $theme_slug => $theme ) {
		// Filter by search query if provided (only if user has typed something).
		if ( ! empty( $query_lower ) ) {
			$name_lower = strtolower( $theme->get( 'Name' ) );
			$desc_lower = strtolower( $theme->get( 'Description' ) );

			if ( strpos( $name_lower, $query_lower ) === false && strpos( $desc_lower, $query_lower ) === false ) {
				continue;
			}
		}

		$is_active = ( $theme_slug === $current_theme );

		$screenshot = $theme->get_screenshot();
		$icon       = $screenshot ? $screenshot : 'dashicons-admin-appearance';

		$results[] = array(
			'id'                   => 'activate-theme-' . $theme_slug,
			'title'                => $theme->get( 'Name' ) . ( $is_active ? ' ' . __( '(Active)', 'commandify' ) : '' ),
			'description'          => sprintf(
				/* translators: 1: Version number, 2: Author name */
				__( 'Version %1$s • %2$s', 'commandify' ),
				$theme->get( 'Version' ),
				$theme->get( 'Author' )
			),
			'type'                 => 'action',
			'icon'                 => $icon,
			'category'             => 'themes',
			'has_dynamic_callback' => true,
			/* translators: %s: Plugin name or theme name */
			'loading_message'      => sprintf( __( 'Activating %s...', 'commandify' ), $theme->get( 'Name' ) ),
			'disabled'             => $is_active,
		// Disable the currently active theme
		);
	}//end foreach

	return $results;
}
