<?php
namespace WpRigel\Commandify;

defined( 'ABSPATH' ) || exit;

/**
 * Register user commands
 * Hooks into commandify_register_commands at priority 40 (after other commands)
 */
add_action(
	'commandify_register_commands',
	function () {
		commandify_register_command(
			array(
				'id'          => 'user-preferences',
				'title'       => __( 'User Preferences', 'commandify' ),
				'description' => __( 'Customize your command palette preferences', 'commandify' ),
				'category'    => 'settings',
				'keywords'    => array( 'settings', 'preferences', 'customize', 'config', 'keyboard', 'shortcut' ),
				'icon'        => 'dashicons-admin-users',
				'type'        => 'submenu',
				'callback'    => __NAMESPACE__ . '\\get_user_preferences_callback',
			)
		);
	},
	40
);

function get_user_preferences_callback() {
	$settings        = Settings::get_instance();
	$global_settings = $settings->get_settings();
	$effective_prefs = Settings::get_effective_preferences();

	$actions = array();

	// Keyboard Shortcut.
	$actions[] = array(
		'id'          => 'set-keyboard-shortcut',
		'title'       => __( 'Change Keyboard Shortcut', 'commandify' ),
		'description' => sprintf(
			/* translators: 1: Current keyboard shortcut, 2: Default keyboard shortcut */
			__( 'Current: %1$s (Default: %2$s)', 'commandify' ),
			$effective_prefs['keyboard_shortcut'],
			$global_settings['keyboard_shortcut']
		),
		'category'    => 'settings',
		'icon'        => 'dashicons-keyboard-hide',
		'type'        => 'action',
		'callback'    => function () {
			return array(
				'success' => true,
				'message' => __( 'Feature coming soon: Interactive keyboard shortcut picker', 'commandify' ),
			);
		},
	);

	// Results Per Page.
	$actions[] = array(
		'id'          => 'set-results-per-page',
		'title'       => __( 'Change Results Per Page', 'commandify' ),
		'description' => sprintf(
			/* translators: 1: Current results per page, 2: Default results per page */
			__( 'Current: %1$d (Default: %2$d)', 'commandify' ),
			$effective_prefs['results_per_page'],
			$global_settings['results_per_page']
		),
		'category'    => 'settings',
		'icon'        => 'dashicons-list-view',
		'type'        => 'submenu',
		'actions'     => array_map(
			function ( $value ) use ( $effective_prefs ) {
				return array(
					'id'       => "set-results-$value",
					/* translators: %d: Number of results to show */
					'title'    => sprintf( __( 'Show %d results', 'commandify' ), $value ),
					'category' => 'settings',
					'icon'     => 'dashicons-yes-alt',
					'type'     => 'action',
					'selected' => $effective_prefs['results_per_page'] === $value,
					'callback' => function () use ( $value ) {
						Settings::update_user_preferences( array( 'results_per_page' => $value ) );
						return array(
							'success' => true,
							/* translators: %d: Number of results */
							'message' => sprintf( __( 'Results per page set to %d', 'commandify' ), $value ),
						);
					},
				);
			},
			array( 10, 20, 30, 50, 75, 100 )
		),
	);

	// Debounce Delay.
	$actions[] = array(
		'id'          => 'set-debounce-delay',
		'title'       => __( 'Change Search Debounce Delay', 'commandify' ),
		'description' => sprintf(
			/* translators: 1: Current debounce delay in milliseconds, 2: Default debounce delay */
			__( 'Current: %1$dms (Default: %2$dms)', 'commandify' ),
			$effective_prefs['debounce_delay'],
			$global_settings['debounce_delay']
		),
		'category'    => 'settings',
		'icon'        => 'dashicons-clock',
		'type'        => 'submenu',
		'actions'     => array_map(
			function ( $value ) use ( $effective_prefs ) {
				/* translators: %d: Delay in milliseconds */
				$label = 0 === $value ? __( 'Instant', 'commandify' ) : sprintf( __( '%dms delay', 'commandify' ), $value );
					return array(
						'id'       => "set-debounce-$value",
						'title'    => $label,
						'category' => 'settings',
						'icon'     => 'dashicons-yes-alt',
						'type'     => 'action',
						'selected' => $value === $effective_prefs['debounce_delay'],
						'callback' => function () use ( $value ) {
							Settings::update_user_preferences( array( 'debounce_delay' => $value ) );
							return array(
								'success' => true,
								/* translators: %d: Delay in milliseconds */
								'message' => sprintf( __( 'Debounce delay set to %dms', 'commandify' ), $value ),
							);
						},
					);
			},
			array( 0, 50, 100, 150, 200, 300, 500 )
		),
	);

	// Reset to defaults.
	$actions[] = array(
		'id'          => 'reset-preferences',
		'title'       => __( 'Reset to Default Settings', 'commandify' ),
		'description' => __( 'Reset all your preferences to global defaults', 'commandify' ),
		'category'    => 'settings',
		'icon'        => 'dashicons-image-rotate',
		'type'        => 'action',
		'confirm'     => array(
			'title'   => __( 'Reset Preferences?', 'commandify' ),
			'message' => __( 'This will reset all your preferences to the global defaults. This action cannot be undone.', 'commandify' ),
		),
		'callback'    => function () {
			$user_id = get_current_user_id();
			delete_user_meta( $user_id, Settings::USER_PREFS_KEY );
			return array(
				'success' => true,
				'message' => __( 'All preferences reset to defaults', 'commandify' ),
			);
		},
	);

	return array(
		'actions' => $actions,
	);
}

/**
 * Register settings command
 * Extends the commandify_register_commands hook
 */
add_action(
	'commandify_register_commands',
	function () {
		commandify_register_command(
			array(
				'id'          => 'open-settings',
				'title'       => __( 'Plugin Settings', 'commandify' ),
				'description' => __( 'Open Commandify plugin settings page', 'commandify' ),
				'category'    => 'settings',
				'keywords'    => array( 'settings', 'plugin', 'admin', 'config' ),
				'icon'        => 'dashicons-admin-settings',
				'type'        => 'navigation',
				'callback'    => admin_url( 'options-general.php?page=commandify-settings' ),
			)
		);
	},
	40
);
